import type { Express } from "express";
import { createServer, type Server } from "http";
import { storage } from "./storage";
import { insertLeadSchema, insertContactSchema, insertDealSchema } from "@shared/schema";
import { z } from "zod";
import { setupAuth, registerAuthRoutes, isAuthenticated } from "./replit_integrations/auth";

export async function registerRoutes(
  httpServer: Server,
  app: Express
): Promise<Server> {
  // Auth setup (MUST be before other routes)
  await setupAuth(app);
  registerAuthRoutes(app);

  // Pipeline Stages
  app.get("/api/pipeline-stages", isAuthenticated, async (req, res) => {
    const stages = await storage.getPipelineStages();
    res.json(stages);
  });

  // Leads
  app.get("/api/leads", isAuthenticated, async (req, res) => {
    const leads = await storage.getLeads();
    res.json(leads);
  });

  app.post("/api/leads", async (req, res) => {
    const result = insertLeadSchema.safeParse(req.body);
    if (!result.success) {
      return res.status(400).json({ error: result.error });
    }
    const lead = await storage.createLead(result.data);
    res.json(lead);
  });

  app.patch("/api/leads/:id", isAuthenticated, async (req, res) => {
    const lead = await storage.updateLead(req.params.id as string, req.body);
    res.json(lead);
  });

  app.post("/api/leads/:id/convert", isAuthenticated, async (req, res) => {
    const lead = await storage.getLead(req.params.id as string);
    if (!lead) return res.status(404).send("Lead not found");

    const contact = await storage.createContact({
      firstName: lead.firstName,
      lastName: lead.lastName,
      email: lead.email,
      phone: lead.phone,
      company: lead.company,
      position: "",
      notes: lead.notes,
      source: "lead_conversion"
    });

    await storage.updateLead(lead.id, { status: "converted" });
    res.json(contact);
  });

  // Contacts
  app.get("/api/contacts", isAuthenticated, async (req, res) => {
    const contacts = await storage.getContacts();
    res.json(contacts);
  });

  app.post("/api/contacts", isAuthenticated, async (req, res) => {
    const result = insertContactSchema.safeParse(req.body);
    if (!result.success) {
      return res.status(400).json({ error: result.error });
    }
    const contact = await storage.createContact(result.data);
    res.json(contact);
  });

  app.patch("/api/contacts/:id", isAuthenticated, async (req, res) => {
    const contact = await storage.updateContact(req.params.id as string, req.body);
    res.json(contact);
  });

  app.delete("/api/contacts/:id", isAuthenticated, async (req, res) => {
    await storage.deleteContact(req.params.id as string);
    res.sendStatus(204);
  });

  // Deals
  app.get("/api/deals", isAuthenticated, async (req, res) => {
    const deals = await storage.getDeals();
    res.json(deals);
  });

  app.post("/api/deals", isAuthenticated, async (req, res) => {
    const result = insertDealSchema.safeParse(req.body);
    if (!result.success) {
      return res.status(400).json({ error: result.error });
    }
    const deal = await storage.createDeal(result.data);
    res.json(deal);
  });

  app.patch("/api/deals/:id", isAuthenticated, async (req, res) => {
    const deal = await storage.updateDeal(req.params.id as string, req.body);
    res.json(deal);
  });

  app.delete("/api/deals/:id", isAuthenticated, async (req, res) => {
    await storage.deleteDeal(req.params.id as string);
    res.sendStatus(204);
  });

  return httpServer;
}
