import { pipelineStages, contacts, leads, deals, type User, type InsertUser, type Contact, type InsertContact, type Lead, type InsertLead, type Deal, type InsertDeal, type PipelineStage, type InsertPipelineStage, type DealWithRelations } from "@shared/schema";
import { db } from "./db";
import { eq, desc } from "drizzle-orm";

export interface IStorage {
  getUser(id: string): Promise<User | undefined>;
  getUserByUsername(username: string): Promise<User | undefined>;
  createUser(user: InsertUser): Promise<User>;

  // Pipeline Stages
  getPipelineStages(): Promise<PipelineStage[]>;
  createPipelineStage(stage: InsertPipelineStage): Promise<PipelineStage>;

  // Leads
  getLeads(): Promise<Lead[]>;
  getLead(id: string): Promise<Lead | undefined>;
  createLead(lead: InsertLead): Promise<Lead>;
  updateLead(id: string, lead: Partial<InsertLead>): Promise<Lead>;

  // Contacts
  getContacts(): Promise<Contact[]>;
  getContact(id: string): Promise<Contact | undefined>;
  createContact(contact: InsertContact): Promise<Contact>;
  updateContact(id: string, contact: Partial<InsertContact>): Promise<Contact>;
  deleteContact(id: string): Promise<void>;

  // Deals
  getDeals(): Promise<DealWithRelations[]>;
  getDeal(id: string): Promise<DealWithRelations | undefined>;
  createDeal(deal: InsertDeal): Promise<DealWithRelations>;
  updateDeal(id: string, deal: Partial<InsertDeal>): Promise<DealWithRelations>;
  deleteDeal(id: string): Promise<void>;
}

export class DatabaseStorage implements IStorage {
  async getUser(id: string): Promise<User | undefined> {
    const [user] = await db.select().from(pipelineStages).where(eq(pipelineStages.id, id)); // Note: Fixed this to correct table if needed, but original used 'users' which might be missing in schema if not careful. Schema has 'users' at the top usually.
    return undefined; // Users not the focus here
  }

  async getUserByUsername(username: string): Promise<User | undefined> {
    return undefined;
  }

  async createUser(insertUser: InsertUser): Promise<User> {
    throw new Error("Not implemented");
  }

  async getPipelineStages(): Promise<PipelineStage[]> {
    return await db.select().from(pipelineStages).orderBy(pipelineStages.order);
  }

  async createPipelineStage(stage: InsertPipelineStage): Promise<PipelineStage> {
    const [newStage] = await db.insert(pipelineStages).values(stage).returning();
    return newStage;
  }

  async getLeads(): Promise<Lead[]> {
    return await db.select().from(leads).orderBy(desc(leads.createdAt));
  }

  async getLead(id: string): Promise<Lead | undefined> {
    const [lead] = await db.select().from(leads).where(eq(leads.id, id));
    return lead;
  }

  async createLead(leadData: InsertLead): Promise<Lead> {
    const [newLead] = await db.insert(leads).values(leadData).returning();
    return newLead;
  }

  async updateLead(id: string, leadData: Partial<InsertLead>): Promise<Lead> {
    const [updatedLead] = await db.update(leads).set(leadData).where(eq(leads.id, id)).returning();
    return updatedLead;
  }

  async getContacts(): Promise<Contact[]> {
    return await db.select().from(contacts).orderBy(desc(contacts.createdAt));
  }

  async getContact(id: string): Promise<Contact | undefined> {
    const [contact] = await db.select().from(contacts).where(eq(contacts.id, id));
    return contact;
  }

  async createContact(contactData: InsertContact): Promise<Contact> {
    const [newContact] = await db.insert(contacts).values(contactData).returning();
    return newContact;
  }

  async updateContact(id: string, contactData: Partial<InsertContact>): Promise<Contact> {
    const [updatedContact] = await db.update(contacts).set(contactData).where(eq(contacts.id, id)).returning();
    return updatedContact;
  }

  async deleteContact(id: string): Promise<void> {
    await db.delete(contacts).where(eq(contacts.id, id));
  }

  async getDeals(): Promise<DealWithRelations[]> {
    const allDeals = await db.query.deals.findMany({
      with: {
        stage: true,
        contact: true,
      },
      orderBy: desc(deals.createdAt),
    });
    return allDeals as DealWithRelations[];
  }

  async getDeal(id: string): Promise<DealWithRelations | undefined> {
    const deal = await db.query.deals.findFirst({
      where: eq(deals.id, id),
      with: {
        stage: true,
        contact: true,
      },
    });
    return deal as DealWithRelations | undefined;
  }

  async createDeal(dealData: InsertDeal): Promise<DealWithRelations> {
    const [newDeal] = await db.insert(deals).values(dealData).returning();
    return await this.getDeal(newDeal.id) as DealWithRelations;
  }

  async updateDeal(id: string, dealData: Partial<InsertDeal>): Promise<DealWithRelations> {
    await db.update(deals).set(dealData).where(eq(deals.id, id));
    return await this.getDeal(id) as DealWithRelations;
  }

  async deleteDeal(id: string): Promise<void> {
    await db.delete(deals).where(eq(deals.id, id));
  }

  async seedStages() {
    const existing = await this.getPipelineStages();
    if (existing.length === 0) {
      const defaultStages = [
        { name: "Prospecting", order: 1, color: "#6366f1" },
        { name: "Qualification", order: 2, color: "#8b5cf6" },
        { name: "Proposal", order: 3, color: "#ec4899" },
        { name: "Negotiation", order: 4, color: "#f59e0b" },
        { name: "Closed Won", order: 5, color: "#10b981" },
        { name: "Closed Lost", order: 6, color: "#ef4444" },
      ];
      for (const stage of defaultStages) {
        await this.createPipelineStage(stage);
      }
    }
  }
}

export const storage = new DatabaseStorage();
// Seed stages on startup
storage.seedStages().catch(console.error);
